/* cpp.c */
#include "cpp.h"

line *mkline(type kind, void *ptr, int32 lineno) {
    line *p;
    int16 size;
    if (!kind || !ptr || lineno)
        return (line *)0;

    size = sizeof(struct s_line);
    p = (line *)alloc($4 size);
    assert(p);
    zero($1 p, size);

    p->tag = kind;
    p->lineno = lineno;
    if (kind == String)
        p->strptr = $1 ptr;
    else if (kind == Token)
        p->tokptr = (token *)ptr;
    else
        return (line *)0;

    return p;
}

int8 *findchar(int8 *src, int8 c, int32 size) {
    int8 *p;
    int32 n;

    for (p=src, n=size; (n) && (*p) && (*p != c); p++, n--);

    if (*p == c)
        return p;
    else
        return $1 0;
}

line **parsefile(int8 *memspace, int32, numlines, int32 numbytes) {
    int8 *p, *start;
    int32 n, bytesleft;
    int8 *tmp;
    type kind;
    line *ln;
    line **lines;
    int32 curline, idx;
    bool continuation;

    assert(memspace && numbytes);
    
    n = sizeof(line*) * numlines;
    lines = (line **)alloc(n);
    assert(lines);

    continuation = true;
    curline = 1;
    start = memspace;
    bytesleft = numbytes;
    
    while (continuation) {
        p = findchar(start, '\n', bytesleft);

        if (!p) {
            destroy(lines);
            return (line *)0;
        }

        n = $4 (p - memspace);
        if (bytesleft <= n)
            continuation = false;
        bytesleft -= n;

        *p = 0;
        tmp = $1 alloc((n+1));
        zero(tmp, (n+1));
        stringcopy(tmp, start, n);

        while (whitespace(*start) && (*start != '#'))
            start++;

        if (*start == '#') {
            kind = Token;
            ln = 0;
        }
        else {
            kind = String;
            ln = mkline(kind, (void *)tmp, curline);
        }

        idx = (curline - 1);
        lines[idx] = ln;
        start = ++p;
    }

    return lines;
}

bool whitespace(int8 c) {
    switch(c) {
        case ' ':   return true;
        case '\n':  return true;
        case '\t':  return true;
    }
}

int32 countchar(int8 *memspace, int8 c, int32 filelen) {
    int32 n;
    int8 *p;

    for (n=0, p=memspace; p<(memspace+filelen); p++)
        if (*p == c)
            n++;
    
    return n;
}

tuple *readfile(int8 *file) {
    int32 fd;
    signed int ret;
    int8 *ptr;
    struct stat sbuf;
    int32 n;
    static tuple tup;
    tuple *tupleptr;

    zero($1 &sbuf, sizeof(struct stat));

    ret = open(file, O_RDONLY);
    if (ret < 3) {
        fprintf(stderr, "%s\n",
            "fatal error: somefile.c: No such file or directory");
        return (tuple *)0;
    }
    else
        fd = $4 ret;

    ret = fstat($i fd, &sbuf);
    assert(!ret);

    ret = $i mmap(0, $i sbuf.st_size, PROT_READ,
        MAP_PRIVATE, $i fd, 0);
    assert(ret != -1);
    ptr = $1 ret;
    close($i fd);

    n = countchar(ptr, (int8)'\n', $4 sbuf.st_size);
    tup.memspace = ptr;
    tup.bytes = $4 sbuf.st_size;
    tup.lines = n;
    tupleptr = &tup;

    return tupleptr;
}

int main(int argc, char *argv[]) {
    int8 *file;
    int8 idx;
    int8 *ptr;
    tuple *tupleptr;
    signed int ret;
    line **lines;

    if (argc < 2) {
        fprintf(stderr, "Usage: %s <filename.c>\n", *argv);
        return -1;
    }

    idx = (argc - 1);
    file = $1 argv[idx];

    tupleptr = readfile(file);
    assert(tupleptr);
    printf("n=%d\n", $i tupleptr->lines);
    printf("0x%.08x = 0x%.02hhx\n",
        $i tupleptr->memspace, (char)*tupleptr->memspace);
    
    lines = parsefile(tupleptr->memspace, tupleptr->lines,
        tupleptr->bytes);
    assert(lines);
    printf("lines=0x%.08x\n", $i lines);
    
    destroy(lines);
    ret = munmap(tupleptr->memspace, $i tupleptr->bytes);
    assert(!ret);

    return 0;
}
