/* esther.c */
#include <esther.h>

int16 stringlen(int8 *src) {
    int8 *p;
    int16 n;

    for (n=0, p=src; *p; n++, p++);
    return n;
}

bool stringcompare(int8 *a, int8 *b) {
    int8 *aptr, *bptr;
    bool ret;

    ret = true;
    for (aptr=a, bptr=b; *aptr && *bptr; aptr++, bptr++)
        if (*aptr != *bptr) {
            ret = false;
            break;
        }

    return ret;
}

int16 if2idx(int8 *interface) {
    struct ifaddrs *head, *p;
    signed int ret;
    int16 idx;
    bool found_interface;

    ret = getifaddrs(&head);
    if (ret)
        return 0;

    found_interface = false;
    for (p=head, idx=1; p; p = p->ifa_next, idx++)
        if (stringcompare($1 p->ifa_name, interface)) {
            found_interface = true;
            break;
        }
    
    freeifaddrs(head);
    return (!found_interface) ?
            0 :
        idx;
}

int32 setup(int8 *interface, mac *srcmac) {
    int32 s;
    signed int tmp;
    struct timeval tv;
    struct sockaddr_ll sock;

    assert(getuid() == 0);

    tv.tv_sec = Timeout;
    tv.tv_usec = 0;
    
    zero($1 &sock, sizeof(struct sockaddr_ll));
    sock.sll_family = AF_PACKET;
    sock.sll_protocol = htons(ETH_P_ALL);
    // sock.sll_hatype = xxx;
    // sock.sll_pkttype = xxx;
    sock.sll_ifindex = $i if2idx(interface);
    sock.sll_halen = 6;
    copy($1 &sock.sll_addr, $1 srcmac, 6);

    // one = $4 1;
    tmp = socket(AF_PACKET, SOCK_RAW, htons(ETH_P_ALL));
    if (tmp > 2)
        s = $4 tmp;
    else
        s = $4 0;
    
    if (!s)
        return 0;
    
    tmp = bind($i s, (struct sockaddr *)&sock, sizeof(struct sockaddr_ll));
    if (tmp < 0) {
        close($i s);
        return 0;
    }
    
    /*
    setsockopt($i s, SOL_IP, IP_HDRINCL,
        (unsigned int *)&one, sizeof(int32));
    */
     setsockopt($i s, SOL_SOCKET, SO_RCVTIMEO,
        &tv, sizeof(struct timeval));
     setsockopt($i s, SOL_SOCKET, SO_RCVTIMEO,
        &tv, sizeof(struct timeval));
    
    return s;
}

ip *mkip(type kind, const int8 *src, const int8 *dst, int16 id_, int16 *cntptr) {
    int16 id;
    int16 size;
    ip *pkt;

    if (!kind || !src || !dst)
        return (ip *)0;

    if (id_)
        id = id_;
    else
        id = *cntptr++;
    
    size = sizeof(struct s_ip);
    pkt = (ip *)malloc($i size);
    assert(pkt);
    zero($1 pkt, size);

    pkt->kind = kind;
    pkt->id = id;
    pkt->src = inet_addr($c src);
    pkt->dst = inet_addr($c dst);
    pkt->payload.rawpkt = (raw *)0;

    if (!pkt->dst) {
        free(pkt);
        return (ip *)0;
    }

    return pkt;
}

void showether(int8 *ident, ether *e) {
    int16 n;
    ip *pkt;
    int8 *m1, *m2;

    m1 = show(&e->dst);
    m2 = show(&e->src);

    if (!e)
        return;

    if (e->payload) {
        pkt = e->payload;
        n = sizeof(struct s_rawether);

        if (pkt->payload.rawpkt)
            n += sizeof(struct s_rawip)
                + (pkt->kind == L4icmp) ?
                        pkt->payload.icmppkt->size :
                    stringlen($1 pkt->payload.rawpkt);
        else
            n += sizeof(struct s_rawip);

        if (pkt->kind == L4icmp)
            n += sizeof(struct s_rawicmp);
    }
    else
        n = sizeof(struct s_rawether);

    printf("(ether *)%s = {\n", $c ident);
    printf("  size:\t %d bytes total\n", $i n);
    printf("  proto:\t 0x%.04hx\n", $i e->protocol);
    printf("  src:\t %s\n", $c m2);
    printf("  dst:\t %s\n", $c m1);
    printf("}\n");

    free(m1);
    free(m2);

    if (e->payload)
        show(e->payload);

    return;
}

void showip(int8 *ident, ip *pkt) {
    int16 n;

    if (!pkt)
        return;

    if (pkt->payload.rawpkt)
        n = sizeof(struct s_rawip)
            + (pkt->kind == L4icmp) ?
                    pkt->payload.icmppkt->size :
                stringlen($1 pkt->payload.rawpkt);
    else
        n = sizeof(struct s_rawip);

    if (pkt->kind == L4icmp)
        n += sizeof(struct s_rawicmp);

    printf("(ip *)%s = {\n", $c ident);
    printf("  size:\t %d bytes total\n", $i n);
    printf("  kind:\t 0x%.02hhx\n", (char)pkt->kind);
    printf("  id:\t 0x%.02hhx\n", $i pkt->id);
    printf("  src:\t %s\n", $c todotted(pkt->src));
    printf("  dst:\t %s\n", $c todotted(pkt->dst));
    printf("}\n");

    if (pkt->payload.icmppkt)
        show(pkt->payload.icmppkt);

    return;
}

int16 endian16(int16 x) {
    int8 a, b;
    int16 y;

    b = (x & 0x00ff);
    a = ((x & 0xff00)>>8);
    y = (b << 8)
            | a;
        
    return y;
}

int16 checksum(int8 *pkt, int16 size) {
    int16 *p;
    int32 acc, b;
    int16 carry;
    int16 n;
    int16 sum;
    int16 ret;

    acc = 0;
    for (n=size, p = (int16 *)pkt; n; n -= 2, p++) {
        b = *p;
        acc += b;
    }
    carry = ((acc & 0xffff0000)>>16);
    sum = (acc & 0x0000ffff);
    ret = ~(sum+carry);

    return ret;
}

bytestring *evalip(ip *pkt) {
    struct s_rawip rawpkt;
    struct s_rawip *rawptr;
    int16 check;
    int8 *p, *ret;
    int8 protocol;
    int16 lengthle;
    int16 lengthbe;
    int16 size;
    bytestring *bs;

    if (!pkt)
        return (bytestring *)0;
    
    protocol = 0;
    switch(pkt->kind) {
        case L4icmp:
            protocol = 1;
            break;
        
        case L4raw:
            protocol = 0;
            break;
        
        default:
            return (bytestring *)0;
            break;
    }

    rawpkt.checksum = 0;
    rawpkt.dscp = 0;
    rawpkt.dst = pkt->dst;
    rawpkt.ecn = 0;
    rawpkt.flags = 0;
    rawpkt.id = endian16(pkt->id);
    rawpkt.ihl = (sizeof(struct s_rawip)/4);
    
    lengthle = 0;
    if (pkt->payload.rawpkt) {
        lengthle = (rawpkt.ihl*4)
            + (pkt->kind == L4icmp) ?
                    sizeof(struct s_rawicmp)
                    + pkt->payload.icmppkt->size :
                stringlen($1 pkt->payload.rawpkt);
        lengthbe = endian16(lengthle);
        rawpkt.length = lengthbe;
    }
    else
        lengthle = rawpkt.length = (rawpkt.ihl*4);

    rawpkt.offset = 0;
    rawpkt.protocol = protocol;
    rawpkt.src = pkt->src;
    rawpkt.ttl = 250;
    rawpkt.version = 4;

    if (lengthle%2)
        lengthle++;
    
    size = sizeof(struct s_rawip);
    p = $1 malloc($i lengthle);
    ret = p;
    assert(p);
    zero(p, lengthle);
    copy(p, $1 &rawpkt, size);
    p += size;

    if (pkt->payload.rawpkt) {
        // size = (sizeof(struct s_rawicmp) +
        //     pkt->payload->size);
        bs = (pkt->kind == L4icmp) ?
                eval(pkt->payload.icmppkt) :
            eval(pkt->payload.rawpkt);
        if (bs) {
            copy(p, bs->data, bs->size);
            free(bs);
        }
    }

    check = checksum(ret, lengthle);
    rawptr = (struct s_rawip *)ret;
    rawptr->checksum = check;

    return mkbytestring(ret, lengthle);
}

bytestring *evalicmp(icmp *pkt) {
    int8 *p, *ret;
    int16 size;
    struct s_rawicmp rawpkt;
    struct s_rawicmp *rawptr;
    int16 check;

    if (!pkt || !pkt->data)
        return (bytestring *)0;
    
    switch (pkt->kind) {
        case echo:
            rawpkt.type = 8;
            rawpkt.code = 0;

            break;
        
        case echoreply:
            rawpkt.type = 0;
            rawpkt.code = 0;

            break;
        
        default:
            return (bytestring *)0;
            break;
    }

    rawpkt.checksum = 0;
    size = sizeof(struct s_rawicmp) + pkt->size;
    if (size%2)
        size++;

    p = $1 malloc($i size);
    ret = p;
    assert(p);
    zero($1 p, size);

    copy(p, $1 &rawpkt, sizeof(struct s_rawicmp));
    p += sizeof(struct s_rawicmp);
    copy(p, pkt->data, pkt->size);

    check = checksum(ret, size);
    rawptr = (struct s_rawicmp *)ret;
    rawptr->checksum = check;

    return mkbytestring(ret, size);
}

bytestring *evalraw(raw *rawpkt) {
    return mkbytestring($1 rawpkt, stringlen($1 rawpkt));
}

void copy(int8 *dst, int8* src, int16 size) {
    int16 n;
    int8 *sptr, *dptr;

    for (dptr=dst, sptr=src, n=size; n; n--)
        *dptr++ = *sptr++;
    
    return;
}

icmp *mkicmp(type kind, const int8 *data, int16 size) {
    int16 n;
    icmp *p;

    if (!data || !size)
        return (icmp *)0;
    
    n = sizeof(struct s_icmp) + size;
    p = (icmp *)malloc($i n);
    assert(p);
    zero($1 p, n);

    p->kind = kind;
    p->size = size;
    p->data = $1 data;

    return p;
}

void showicmp(int8 *ident, icmp *pkt) {
    struct s_ping *pingptr;

    if (!pkt)
        return;
    
    printf("(icmp *)%s = {\n", $c ident);
    printf("  kind:\t %s\n  size:\t %d bytes of payload\n}\npayload:\n",
        (pkt->kind == echo) ? "echo" : "echo reply",
        $i pkt->size);
    if (pkt->data)
        if ((pkt->kind == echo) || (pkt->kind == echoreply)) {
            pingptr = (struct s_ping *)pkt->data;
            printf("  id:\t %d\n  seq:\t %d\n",
                (unsigned int)endian16(pingptr->id),
                (unsigned int)endian16(pingptr->seq));
        }
    printf("\n");

    return;
}

int8 *showmac(mac *addr) {
    int8 *p;
    int8 a, b, c, d, e, f;
    int64 integer;

    if (!addr)
        return $1 0;

    integer = (addr->addr & 0x0000ffffffffffff);
    f = (integer & 0xff);
    e = ((integer & 0xff00) >> 8);
    d = ((integer & 0xff0000) >> 16);
    c = ((integer & 0xff000000) >> 24);
    b = ((integer & 0xff00000000) >> 32);
    a = ((integer & 0xff0000000000) >> 40);

    p = $1 malloc(16);
    assert(p);
    zero($1 p, 16);
    snprintf($c p, 15,
        "%.02hhx%.02hhx.%.02hhx%.02hhx.%.02hhx%.02hhx",
        f, e, d, c, b, a);
    
    return p;
}

mac *readmacint(int64 input) {
    mac *output;
    int16 size;

    size = sizeof(struct s_mac);
    output = (mac *)malloc($i size);
    assert(output);
    output->addr = (input & 0x0000ffffffffffff);

    return output;
}

mac *readmacstr(int8 *input) {
    int64 integer;
    int8 a, b, c, d, e, f;
    int8 byte, x, y, x_, y_;
    int8 *p;
    bool parity;

    parity = false;
    if (!input)
        return (mac *)0;
    
    p = input;
    parsemac(a);
    parsemac(b);
    parsemac(c);
    parsemac(d);
    parsemac(e);
    parsemac(f);

    integer =
        ((int64)f << 40)
        | ((int64)e << 32)
        | ((int64)d << 24)
        | ((int64)c << 16)
        | ((int64)b << 8)
        | a;

    return readmacint(integer);
}

ether *mkether(ethertype type, mac *dst, mac *src) {
    ether *e;
    int16 size;

    if (!src || !dst)
        return (ether *)0;
    
    size = sizeof(struct s_ether);
    e = (ether *)malloc($i size);
    zero($1 e, size);

    e->protocol = type;
    e->dst = *dst;
    e->src = *src;
    e->payload = (ip *)0;

    return e;
}

bool sendframe(int32 fd, ether *e) {
    bytestring *bs;
    signed int ret;

    if (!fd || !e)
        return false;
    
    show(e);
    bs = eval(e);

    ret = send(fd, $c bs->data, $i bs->size, 0);

    return (ret > 0) ?
            true :
        false;
}

bytestring *evalether(ether *e) {
    int8 *p;
    int16 size;
    struct s_rawether *raw;
    bytestring *bs1, *bs2;

    if (!e)
        return (bytestring *)0;
    
    size = sizeof(struct s_rawether);
    p = $1 malloc($i size);
    assert(p);
    zero($1 p, size);
    raw = (struct s_rawether *)p;

    raw->dst = e->dst;
    raw->src = e->src;
    raw->type = e->protocol;

    bs1 = mkbytestring(p, size);
    if (!bs1)
        return (bytestring *)0;
    
    if (!e->payload)
        return bs1;
    
    bs2 = eval(e->payload);
    if (!bs2) {
        free(bs1->data);
        free(bs1);

        return (bytestring *)0;
    }

    return bs1->merge(bs1, bs2);
}

int main1() {
    mac *m, *m_;
    int8 *str = $1 "0001.38fb.99aa";
    int64 n;

    m = mkmac(str);
    printf("m = '%s'\n",
        show(m));
    printf("0x%.16llx\n",
        (int64)m->addr);
    n = (int64)m->addr;
    m_ = mkmac(n);
    printf("m = '%s'\n",
        show(m_));
    printf("0x%.16llx\n",
        (int64)m_->addr);
 
    return 0;
}

int main2() {
    printf("0x%.02hhx 0x%.02hhx\n",
        (char)hex('a'),
        (char)hex('5')
    );

    return 0;
}

int main3() {
    bytestring *bs1, *bs2;
    int8 *d1, *d2;

    d1 = $1 strdup("Hello ");
    d2 = $1 strdup("world");

    bs1 = mkbytestring(d1, $2 6);
    bs2 = mkbytestring(d2, $2 6);

    bs1 = bs1->merge(bs1, bs2);
    if (!bs1) {
        printf("error\n");
        return -1;
    }

    printf("bs1 = 0x%x\n", $i bs1);
    printf("  size: %d\n", $i bs1->size);
    printf("  data: '%s'\n", $c bs1->data);

    free(bs1);

    return 0;
}

int main4() {
    ether *e;
    mac *src, *dst;
    ip *ippkt;
    int16 cnt;
    int16 *cntptr;
    bytestring *bs;
    icmp *icmppkt;
    struct s_ping *pingpkt;

    cnt = 5;
    cntptr = &cnt;

    src = mkmac($1 "0001.384b.92ab");
    dst = mkmac($1 "ffff.ffff.ffff");
    e = mkether(tIP, dst, src);
    if (!e) {
        printf("error\n");
        return -1;
    }

    ippkt = mkip(L4icmp, $1 "192.168.12.1", $1 "192.168.12.255", 0, cntptr);
    e->payload = ippkt;

    pingpkt = (struct s_ping *)malloc((sizeof(struct s_ping) + 4));
    assert(pingpkt);
    zero($1 pingpkt, (sizeof(struct s_ping) + 4));
    pingpkt->id = 400;
    pingpkt->seq = 2;
    strncpy($c &pingpkt->data, "abc", 3);

    icmppkt = mkicmp(echo, $1 pingpkt, $2 (sizeof(
        struct s_ping) + 4
    ));
    ippkt->payload.icmppkt = icmppkt;

    show(e);
    
    bs = eval(e);
    if (!bs) {
        printf("error\n");
    }
    else
        printhex(bs->data, bs->size, ' ');
    
    if (bs) {
        free(bs->data);
        free(bs);
    }

    free(ippkt);
    free(src);
    free(dst);
    free(e);

    return 0;
}

int main5(int argc, char *argv[]) {
    mac *macsrc, *macdst;
    int8 *ipsrc, *ipdst;
    int8 *msg, *interface;
    int32 fd;
    ether *eth;
    ip *ippkt;
    int16 cnt;
    int16 *cntptr;
    bool ret;

    if (argc < 6)
        usage(argv[0]);

    cnt = 0;
    cntptr = &cnt;

    interface = $1 argv[1];
    macsrc = mkmac($1 argv[2]);
    macdst = mkmac($1 argv[3]);
    ipsrc = $1 argv[4];
    ipdst = $1 argv[5];
    msg = $1 argv[6];

    if (!interface || !macsrc || !macdst || !ipsrc || !ipdst || !msg)
        usage(argv[0]);

    fd = setup(interface, macsrc);
    if (!fd) {
        fprintf(stderr, "socket() failed\n");
        return -2;
    }

    eth = mkether(tIP, macdst, macsrc);
    ippkt = mkip(L4raw, ipsrc, ipdst, 0, cntptr);

    if (!eth || !ippkt) {
        fprintf(stderr, "Unable to parse ether* and/or ip*\n");
        return -3;
    }

    eth->payload = ippkt;
    // ippkt->payload.rawpkt = (raw *)msg;
    ippkt->payload.rawpkt = (raw *)0;
    ret = sendframe(fd, eth);

    printf("ret=%s\n",
        (ret) ?
                "true" :
            "false"
    );

    (void)macdst;
    (void)macdst;
    (void)ipsrc;
    (void)ipdst;
    (void)msg;

    close(fd);
    free(macsrc);
    free(macdst);

    return 0;
}

int main(int argc, char *argv[]) {
    return main5(argc, argv);
}

bytestring *mkbytestring(int8* data, int16 size) {
    int16 n;
    bytestring *bs;

    if (!data || !size)
        return (bytestring *)0;
    
    n = sizeof(struct s_bytestring);
    bs = (bytestring *)malloc($i n);
    assert(bs);
    zero($1 bs, n);

    bs->size = size;
    bs->data = data;
    bs->merge = &bsmerge;

    return bs;
}

static bytestring *bsmerge(bytestring *a, bytestring *b) {
    int16 size;
    int8 *p, *ret;

    if (!a || !b || (!a->size && !b->size))
        return (bytestring *)0;
    if (!a->size)
        return b;
    else if (!b->size)
        return a;
    
    size = (a->size + b->size);
    p = $1 malloc($i size);
    assert(p);
    ret = p;
    zero($1 p, size);
    copy(p, a->data, a->size);
    p += a->size;
    copy(p, b->data, b->size);

    free(a->data);
    free(b->data);
    free(a);
    free(b);

    return mkbytestring(ret, size);
}

int main6() {
    int16 ours, theirs;

    ours = if2idx($1 "ens224");
    theirs = $2 if_nametoindex("ens224");

    printf("ours=\t %d\ntheirs=\t %d\n",
        $i ours, $i theirs);
    
    return 0;
}
