/* xsc256.c */
#define Main
#include <xsc256.h>

state *mkstate(int256 basekey) {
    int128 ke, w;
    int128 *p;
    state *st;
    int16 size;

    p = (int128 *)&basekey;
    ke = *p;
    w = p[1];

    size = sizeof(struct s_state);
    st = (state *)alloc(size);
    assert(p);
    zero($1 p, size);

    p = (int128 *)st->w;
    *p = w;
    st->subkey = gensubkeys(ke);

    return st;
}

roundkey *nextrk(state *st) {
    roundkey *p;

    if (!st)
        return (roundkey *)0;
    
    p = st->subkey;
    st->subkey = st->subkey->next;

    return p;
}

int64 P(int16 input) {
    int16 x;
    int64 y;

    y = 0;
    x = input;

    y = $8 x << 48;
    y |= $8 x << 32;
    y |= $8 x << 16;
    y |= x;

    return y;
}

int192 xbox(int128 input) {
    int8 *x, *y;
    const int8 *iptr;
    int192 output;
    int8 n, tmp, idx;

    x = $1 &input;
    y = $1 &output;
    iptr = Indices;

    for (n=0; n<16; n++) {
        if ((n%2)) {
            idx = *iptr;
            iptr++;
            y[idx] = x[n];
        }
        else {
            //y[*iptr] = y[*++iptr] = x[n];
            tmp = x[n];
            idx = *iptr;
            iptr++;
            y[idx] = tmp;
            idx = *iptr;
            iptr++;
            y[idx] = tmp;
        }
    }

    return output;
}

void showstate(int8 *ident, state *st) {
    int8 n;

    if (!st)
        return;
    
    printf("(state *)%s = {\n  ", ident);
    show(st->subkey);
    printf("  %s->w = { ", ident);
    for (n=0; n<16; n++)
        printf("%.02hhx ", (char)st->w[n]);
    printf("}\n}\n");

    return;
}

void showint256(int8 *ident, int256 input) {
    void *mem;
    int32 *p;
    int8 i;

    printf("%s = 0x", ident);
    mem = $v ($v &input+32);
    for (i=0; i<8; i++) {
        mem -= 4;
        p = (int32 *)mem;
        printf("%.08x", $i *p);
    }
    printf("\n");

    return;
}

void showsemiword(int8 *ident, semiword input) {
    printf("%s = 0x%.01hhx%.01hhx%.01hhx\n", $c ident,
        (char)input.x, (char)input.y, (char)input.z);

    return;
}

void showint64(int8 *ident, int64 input) {
    printf("%s = 0x%.16llx\n", $c ident, (long long int)input);

    return;
}

void showint128(int8 *ident, int128 input) {
    void *mem;
    int32 *p;
    int8 i;

    printf("%s = 0x", ident);
    mem = $v ($v &input+16);
    for (i=0; i<4; i++) {
        mem -= 4;
        p = (int32 *)mem;
        printf("%.08x", $i *p);
    }
    printf("\n");

    return;
}

void showint192(int8 *ident, int192 input) {
    void *mem;
    int32 *p;
    int8 i;

    printf("%s = 0x", ident);
    mem = $v ($v &input+24);
    for (i=0; i<6; i++) {
        mem -= 4;
        p = (int32 *)mem;
        printf("%.08x", $i *p);
    }
    printf("\n");

    return;
}

int192 sboxes(int192 input) {
    int8 n;
    int192 output;
    semiword *x, *y;

    output.x = 0;
    x = (semiword *)&input;
    y = (semiword *)&output;

    for (n=0; n<16; n++)
            y[n] = sbox(x[n]);

    return output;
}

void showroundkey(int8 *id, roundkey *rk) {
    assert(rk);

    printf("%s = {\n", $c id);
    printf("    id=%d\n", rk->id);
    printf("    rc=0x%.16llx\n  ", $8 rk->rc);
    show(rk->subkey);
    printf("}\n");

    return;
}

semiword sbox(semiword input) {
    return Sbox[input.z][input.y][input.x];
}

int128 cbox(int192 input) {
    int128 output;
    int8 *x, *y;
    int8 n, idx, tmp;

    x = $1 &input;
    y = $1 &output;

    for (idx=n=0; n<8; n++) {
        tmp = x[(idx+1)];
         y[idx] = x[idx] & tmp;
         idx++;
        tmp = x[(idx+1)];
         y[idx] = x[idx] | tmp;
         idx++;
    }

    return output;
}

int128 f(int128 input) {
    int128 output;
    output = cbox(sboxes(xbox(input)));

    return output;
}

roundkey *mkroundkey(int8 id, int128 subkey, int64 rc) {
    int16 size;
    roundkey *p;

    size = sizeof(struct s_roundkey);
    p = alloc(size);
    assert(p);
    zero($1 p, size);

    p->id = id;
    p->subkey = subkey;
    p->rc = rc;
    p->next = (roundkey *)0;

    return p;
}

int128 mksubkey(int8 id, int128 subkey, int64 rc) {
    return subkey;
}

roundkey *gensubkeys(int128 key) {
    roundkey *p, *last, *first;
    int64 rc;
    int8 x;

    x = 0;
    first = mkroundkey(x, key, RCs[x]);
    x++;

    for (last = first; x<16; x++) {
        rc = P(RCs[x]);
        p = mkroundkey(x, g(last->subkey,rc), rc);
        last->next = p;
        last = p;
    }
    last->next = first;

    return first;
}

void zero(int8* x, int16 size) {
    int16 n;
    int8 *p;

    for (p=x, n=size; n; n--, p++)
        *p = 0;
    
    return;
}

void mksbox() {
    int8 x,y,z;

    for (x=0; x<=0x0f; x++)
        for (y=0; y<=0x0f; y++)
            for (z=0; z<=0x0f; z++)
                Sbox[z][y][x] = (semiword){.x=z, .y=y, .z=x};
    
    Sbox[0][0][0] = (semiword){5,5,5};
}

#define _rotl(b)    int ## b rotl ## b (int ## b,int8); \
int ## b rotl ## b (int ## b input, int8 n) { \
    int ## b x=input; for (int8 i=0; i<n; i++) x=rotl1((b),x); return x; \
}
_rotl(8);
_rotl(16);
_rotl(32);
_rotl(64);
_rotl(128);

#define _rotr(b) int ## b rotr ## b (int ## b,int8); \
int ## b rotr ## b (int ## b input, int8 n) { \
    int ## b x=input; for (int8 i=0; i<n; i++) x=rotr1((b),x); return x; \
}
_rotr(8);
_rotr(16);
_rotr(32);
_rotr(64);
_rotr(128);

int128 g(int128 x, int64 rc) {
    int128 y;
    semiword *p;
    void *mem;

    y = rotl(x, 7);
    y = y * rc;
    
    y = rotr(y, 4);
    mem = $v &y;
    mem += 14;
    p = (semiword *)mem;
    *p = sbox(*p);
    y = rotl(y, 4);

    return y;
}

export state *xsc256init(int256 basekey) {
    state *st;

    mksbox();
    st = mkstate(basekey);
    rnds(st, 62);

    return st;
}

export unsigned char xsc256byte(state *st) {
    int8 idx;
    unsigned char byte;

    rnds(st, 2);
    idx = grabidx(st->w[0]);
    byte = (unsigned char)st->w[idx];

    return byte;
}

export unsigned char *xsc256encrypt(state *st, unsigned char *input,
    unsigned short int len) {
        int16 size, n;
        int8 *buf, *yptr;
        unsigned char *xptr, *ret;

        assert(st && input && len);
        size = $2 len;
        buf = $1 alloc(size);
        assert(buf);
        zero($1 buf, size);

        xptr = input;
        yptr = buf;

        for (n=size; n; n--, xptr++, yptr++)
            *yptr = *xptr ^ xsc256byte(st);
        ret = (unsigned char *)buf;

        return ret;
}

export void xsc256uninit(state *st) {
    int8 n;
    roundkey *rk;

    if (!st)
        return;
    
    for (n=0, rk=nextrk(st); (n<16) && rk; rk=nextrk(st), n++)
        destroy(rk);
    free(st);

    return;
}

export int256 xsc256kdf(unsigned char *input) {
    int256 *ke;
    int8 hash[256];
    SHA2_CTX ctx;
    int16 inputlength;
    int8 n;

    assert(input != 0);
    inputlength = $2 strlen($c input);
    assert(inputlength <= 256);
    zero($1 &hash, 256);
    strncpy($c hash, $c input, $i inputlength);

    for (n=0; n<64; n++) {
        zero($1 &ctx, sizeof(SHA2_CTX));
        SHA256Init(&ctx);
        SHA256Update(&ctx, hash, $i inputlength);
        SHA256Final(hash, &ctx);
    }
    ke = (int256 *)&hash;

    return *ke;
}

export void encryptfile_(int256 ke) {
    int8 n;
    int8 *p;
    int8 buf[1024];
    signed int ret;
    state *st;

    st = xsc256init(ke);
    do {
        zero($1 &buf, 1024);
        ret = read(0, $c buf, 1023);
        if (ret < 1)
            break;
        else
            n = (int8)ret;
        
        p = xsc256encrypt(st, buf, n);
        if (!p)
            break;
        write(1, $c p, $i n);
        destroy(p);
    } while(true);
    xsc256uninit(st);

    return;
}

void rnd(state *st) {
    int128 w;
    int128 *wptr;

    assert(st);
    wptr = (int128 *)&st->w;
    w = *wptr;

    w = rotr(w, 7);
    w = f(w);
    w ^= nextrk(st)
        -> subkey;
    *wptr = w;
    // show(w);

    return;
}

void main1() {
    int8 n;
    int8 *p;
    int128 x;

    x = 0;
    for (n=0, p=$1 &x; n<16; n++, p++)
        *p = n;

    for (n=0; n<64; n++)
        x = f(x);
    show(x);

    return;
}

void main2() {
    int128 x;
    roundkey *rk;
    int8 n;

    x = 3;
    rk = gensubkeys(x);
    for (n=0; n<16; n++, rk = rk->next)
        show(rk);

    return;
}

void main3() {
    state *st;
    int256 x;
    int8 *p;
    int8 n;
    unsigned char byte;
    // roundkey *rk;

    x = 0;
    for (n=0, p=$1 &x; n<32; n++, p++)
        *p = n;

    // show(x);
    st = xsc256init(x);
    // show(st);
    // for (n=0, rk=nextrk(st); n<17; n++, rk = nextrk(st))  
        // show(rk);

    for (n=0; n<12; n++) {
        byte = xsc256byte(st);
        //show(st);
        printf("%.02hhx\n", (char)byte);
    }
    xsc256uninit(st);

    return;

}

void main4(unsigned char *key) {
    int256 x;

    x = 3423948239482;
    encryptfile(x);

    return;
}

int main(int argc, char *argv[]) {
    if (argc < 2) {
        fprintf(stderr, "Usage: cat input | %s <key> >> output.txt\n",
            *argv);
        return -1;
    }
    //mksbox();
    main4((unsigned char *)argv[1]);

    return 0;
}
